/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package geb.test

import org.openqa.selenium.htmlunit.HtmlUnitDriver
import spock.lang.Specification
import spock.lang.TempDir

class GebTestManagerSpec extends Specification {

    @TempDir
    File classpathResourceDir

    def "GebTestManager by default executes GebConfig script only once even if multiple browser instances are requested"() {
        given:
        // language=Groovy
        def gebConfig = """
            def propertyName = "geb.test.GebTestManagerSpec.hasExecuted"

            if (System.getProperty(propertyName)) {
                throw new Exception("GebConfig loaded more than once!")
            }

            System.setProperty(propertyName, true.toString())

            driver = "htmlunit"
        """

        and:
        def gebTestManager = new GebTestManagerBuilder().build()

        when:
        def browser = withOverriddenGebConfig(gebConfig) {
            gebTestManager.browser
            gebTestManager.resetBrowser()
            gebTestManager.browser
        }

        then:
        noExceptionThrown()

        and:
        browser.driver instanceof HtmlUnitDriver
    }

    private <T> T withOverriddenGebConfig(String gebConfig, Closure<T> closure) {
        new File(classpathResourceDir, "GebConfig.groovy").text = gebConfig

        def currentThread = Thread.currentThread()
        def originalContextClassLoader = currentThread.contextClassLoader
        try {
            currentThread.contextClassLoader = new URLClassLoader(classpathResourceDir.toURI().toURL())
            closure.call()
        } finally {
            currentThread.contextClassLoader = originalContextClassLoader
        }
    }
}
