/*
 * Decompiled with CFR 0.152.
 */
package org.apache.netbeans.nbpackage.macos;

import java.io.IOException;
import java.lang.invoke.CallSite;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.netbeans.nbpackage.AbstractPackagerTask;
import org.apache.netbeans.nbpackage.Architecture;
import org.apache.netbeans.nbpackage.ExecutionContext;
import org.apache.netbeans.nbpackage.FileUtils;
import org.apache.netbeans.nbpackage.NBPackage;
import org.apache.netbeans.nbpackage.StringUtils;
import org.apache.netbeans.nbpackage.macos.MacOS;

class AppBundleTask
extends AbstractPackagerTask {
    static final String ARCH_X86_64 = "x86_64";
    static final String ARCH_ARM64 = "arm64";
    static final String ARCH_UNIVERSAL = "universal";
    private static final String DEFAULT_JAR_INTERNAL_BIN_GLOB = "**/*.{dylib,jnilib}";
    private static final String NATIVE_BIN_FILENAME = "nativeBinaries";
    private static final String JAR_BIN_FILENAME = "jarBinaries";
    private static final String ENTITLEMENTS_FILENAME = "sandbox.plist";
    private static final String LAUNCHER_SRC_DIRNAME = "macos-launcher-src";
    private String bundleName;
    private String bundleArch;

    AppBundleTask(ExecutionContext context) {
        super(context);
    }

    @Override
    protected void checkPackageRequirements() throws Exception {
        String[] cmds = this.context().getValue(MacOS.CODESIGN_ID).isEmpty() ? new String[]{"swift"} : new String[]{"swift", "codesign"};
        this.validateTools(cmds);
    }

    @Override
    protected void customizeImage(Path image) throws Exception {
        Path bundle = image.resolve(this.bundleName() + ".app");
        Path contents = bundle.resolve("Contents");
        Path resources = contents.resolve("Resources");
        String execName = this.findLauncher(resources.resolve("APPDIR").resolve("bin")).getFileName().toString();
        Files.move(resources.resolve("APPDIR"), resources.resolve(execName), new CopyOption[0]);
        Files.createDirectory(contents.resolve("MacOS"), new FileAttribute[0]);
        this.setupIcons(resources, execName);
        this.setupInfo(contents, execName);
        this.setupLauncherSource(image);
    }

    @Override
    protected void finalizeImage(Path image) throws Exception {
        Path bundle = image.resolve(this.bundleName() + ".app");
        this.setupSigningConfiguration(image, bundle);
    }

    @Override
    protected Path buildPackage(Path image) throws Exception {
        Path bundle = image.resolve(this.bundleName() + ".app");
        String execName = FileUtils.find(bundle, "Contents/Resources/*/bin/*").stream().filter(path -> !path.toString().endsWith(".exe")).findFirst().map(path -> path.getFileName().toString()).orElseThrow();
        String arch = this.bundleArch();
        Path launcher = this.compileLauncher(image.resolve(LAUNCHER_SRC_DIRNAME), arch);
        Files.copy(launcher, bundle.resolve("Contents").resolve("MacOS").resolve(execName), StandardCopyOption.COPY_ATTRIBUTES);
        String signID = this.context().getValue(MacOS.CODESIGN_ID).orElse("");
        if (signID.isBlank()) {
            this.context().warningHandler().accept(MacOS.MESSAGES.getString("message.nocodesignid"));
            return bundle;
        }
        Path entitlements = image.resolve(ENTITLEMENTS_FILENAME);
        this.signBinariesInJARs(image, entitlements, signID);
        this.signNativeBinaries(image, entitlements, signID);
        this.codesign(bundle, entitlements, signID);
        return bundle;
    }

    @Override
    protected String calculateImageName(Path input) throws Exception {
        String arch = this.bundleArch();
        if (ARCH_UNIVERSAL.equals(arch)) {
            return super.calculateImageName(input) + "-macOS-app";
        }
        return super.calculateImageName(input) + "-macOS-" + arch + "-app";
    }

    @Override
    protected Path calculateAppPath(Path image) throws Exception {
        return image.resolve(this.bundleName() + ".app").resolve("Contents").resolve("Resources").resolve("APPDIR");
    }

    @Override
    protected Path calculateRuntimePath(Path image, Path application) throws Exception {
        return image.resolve(this.bundleName() + ".app").resolve("Contents").resolve("Home");
    }

    @Override
    protected Path calculateRootPath(Path image) throws Exception {
        return image.resolve(this.bundleName() + ".app");
    }

    String bundleName() {
        if (this.bundleName == null) {
            String name = this.sanitize(this.context().getValue(NBPackage.PACKAGE_NAME).orElseThrow());
            if (name.length() > 15) {
                name = name.substring(0, 16);
            }
            this.bundleName = name;
        }
        return this.bundleName;
    }

    String bundleArch() {
        if (this.bundleArch == null) {
            this.bundleArch = this.context().getValue(NBPackage.PACKAGE_ARCH).or(() -> this.context().getValue(MacOS.ARCH)).map(arch -> {
                if (Architecture.X86_64.isSynonym((String)arch)) {
                    return ARCH_X86_64;
                }
                if (Architecture.AARCH64.isSynonym((String)arch)) {
                    return ARCH_ARM64;
                }
                if (!arch.equalsIgnoreCase(ARCH_UNIVERSAL)) {
                    this.context().warningHandler().accept(MacOS.MESSAGES.getString("message.unknownarch"));
                }
                return ARCH_UNIVERSAL;
            }).orElseGet(() -> {
                Optional<Path> runtime = this.context().getValue(NBPackage.PACKAGE_RUNTIME);
                if (runtime.isPresent()) {
                    return Architecture.detectFromPath(runtime.get()).map(a -> switch (a) {
                        default -> throw new IncompatibleClassChangeError();
                        case Architecture.AARCH64 -> ARCH_ARM64;
                        case Architecture.X86_64 -> ARCH_X86_64;
                    }).orElseGet(() -> {
                        this.context().warningHandler().accept(MacOS.MESSAGES.getString("message.unknownarch"));
                        return ARCH_UNIVERSAL;
                    });
                }
                return ARCH_UNIVERSAL;
            });
        }
        return this.bundleArch;
    }

    void validateTools(String ... tools) throws Exception {
        if (this.context().isVerbose()) {
            this.context().infoHandler().accept(MessageFormat.format(MacOS.MESSAGES.getString("message.validatingtools"), Arrays.toString(tools)));
        }
        for (String tool : tools) {
            if (this.context().exec(List.of("which", tool)) == 0) continue;
            throw new IllegalStateException(MessageFormat.format(MacOS.MESSAGES.getString("message.missingtool"), tool));
        }
    }

    String sanitize(String name) {
        return name.replaceAll("[\\\\/:*?\"<>|]", "_");
    }

    private String sanitizeBundleID(String name) {
        return name.replaceAll("[^a-zA-Z0-9-\\.]", "-");
    }

    private Path findLauncher(Path binDir) throws IOException {
        try (Stream<Path> files = Files.list(binDir);){
            Path path = files.filter(f -> !f.getFileName().toString().endsWith(".exe")).findFirst().orElseThrow(IOException::new);
            return path;
        }
    }

    private void setupIcons(Path resources, String execName) throws IOException {
        Path icnsFile = this.context().getValue(MacOS.ICON_PATH).orElse(null);
        Path dstFile = resources.resolve(execName + ".icns");
        if (icnsFile != null) {
            Files.copy(icnsFile, dstFile, new CopyOption[0]);
        } else {
            Files.copy(this.getClass().getResourceAsStream("/org/apache/netbeans/nbpackage/apache-netbeans.icns"), dstFile, new CopyOption[0]);
        }
    }

    private void setupInfo(Path contents, String execName) throws IOException {
        String template = MacOS.INFO_TEMPLATE.load(this.context());
        Map<String, CallSite> tokenMap = Map.of("BUNDLE_NAME", this.bundleName(), "BUNDLE_DISPLAY", this.context().getValue(NBPackage.PACKAGE_NAME).orElseThrow(), "BUNDLE_VERSION", this.context().getValue(NBPackage.PACKAGE_VERSION).orElseThrow(), "BUNDLE_EXEC", execName, "BUNDLE_ID", this.context().getValue(MacOS.BUNDLE_ID).orElse(this.sanitizeBundleID(this.bundleName())), "BUNDLE_ICON", execName + ".icns");
        String info = StringUtils.replaceTokens(template, tokenMap);
        Files.writeString(contents.resolve("Info.plist"), (CharSequence)info, StandardOpenOption.CREATE_NEW);
    }

    private void setupLauncherSource(Path image) throws IOException {
        Path launcherProject = image.resolve(LAUNCHER_SRC_DIRNAME);
        Files.createDirectories(launcherProject, new FileAttribute[0]);
        Path sourceDir = launcherProject.resolve("Sources").resolve("AppLauncher");
        Files.createDirectories(sourceDir, new FileAttribute[0]);
        String packageSwift = MacOS.LAUNCHER_PACKAGE_TEMPLATE.load(this.context());
        String mainSwift = MacOS.LAUNCHER_TEMPLATE.load(this.context());
        Files.writeString(launcherProject.resolve("Package.swift"), (CharSequence)packageSwift, StandardOpenOption.CREATE_NEW);
        Files.writeString(sourceDir.resolve("main.swift"), (CharSequence)mainSwift, StandardOpenOption.CREATE_NEW);
    }

    private void setupSigningConfiguration(Path image, Path bundle) throws IOException {
        Files.writeString(image.resolve(ENTITLEMENTS_FILENAME), (CharSequence)MacOS.ENTITLEMENTS_TEMPLATE.load(this.context()), StandardOpenOption.CREATE_NEW);
        List<Path> nativeBinaries = FileUtils.find(bundle, this.context().getValue(MacOS.SIGNING_FILES).orElseThrow());
        Files.writeString(image.resolve(NATIVE_BIN_FILENAME), (CharSequence)nativeBinaries.stream().map(path -> image.relativize((Path)path)).map(Path::toString).collect(Collectors.joining("\n", "", "\n")), StandardOpenOption.CREATE_NEW);
        List<Path> jarBinaries = FileUtils.find(bundle, this.context().getValue(MacOS.SIGNING_JARS).orElseThrow());
        Files.writeString(image.resolve(JAR_BIN_FILENAME), (CharSequence)jarBinaries.stream().map(path -> image.relativize((Path)path)).map(Path::toString).collect(Collectors.joining("\n", "", "\n")), StandardOpenOption.CREATE_NEW);
    }

    private Path compileLauncher(Path launcherProject, String arch) throws IOException, InterruptedException {
        ProcessBuilder pb = switch (arch) {
            case ARCH_X86_64 -> new ProcessBuilder("swift", "build", "--configuration", "release", "--arch", ARCH_X86_64);
            case ARCH_ARM64 -> new ProcessBuilder("swift", "build", "--configuration", "release", "--arch", ARCH_ARM64);
            default -> new ProcessBuilder("swift", "build", "--configuration", "release", "--arch", ARCH_ARM64, "--arch", ARCH_X86_64);
        };
        pb.directory(launcherProject.toFile());
        this.context().exec(pb);
        List<Path> output = FileUtils.find(launcherProject.resolve(".build"), "**/{R,r}elease/AppLauncher");
        if (output.isEmpty()) {
            throw new IOException(launcherProject.toString());
        }
        return output.get(0);
    }

    private void signBinariesInJARs(Path image, Path entitlements, String id) throws IOException {
        Path jarFiles = image.resolve(JAR_BIN_FILENAME);
        if (!Files.exists(jarFiles, new LinkOption[0])) {
            return;
        }
        List jars = Files.readString(jarFiles).lines().filter(l -> !l.isBlank()).map(x$0 -> Path.of(x$0, new String[0])).map(image::resolve).collect(Collectors.toList());
        for (Path jar : jars) {
            FileUtils.processJarContents(jar, DEFAULT_JAR_INTERNAL_BIN_GLOB, (file, path) -> {
                this.codesign(file, entitlements, id);
                return true;
            });
        }
    }

    private void signNativeBinaries(Path image, Path entitlements, String id) throws IOException {
        Path nativeFiles = image.resolve(NATIVE_BIN_FILENAME);
        if (!Files.exists(nativeFiles, new LinkOption[0])) {
            return;
        }
        List files = Files.readString(nativeFiles).lines().filter(l -> !l.isBlank()).map(x$0 -> Path.of(x$0, new String[0])).map(image::resolve).collect(Collectors.toList());
        for (Path file : files) {
            this.codesign(file, entitlements, id);
        }
    }

    private void codesign(Path file, Path entitlements, String id) throws IOException {
        try {
            this.context().exec("codesign", "--force", "--timestamp", "--options=runtime", "--entitlements", entitlements.toString(), "-s", id, "-v", file.toString());
        }
        catch (InterruptedException ex) {
            throw new IOException(ex);
        }
    }
}

