/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.bean.html5;

import java.util.*;

import org.apache.juneau.annotation.*;

/**
 * DTO for an HTML <a class="doclink" href="https://www.w3.org/TR/html5/forms.html#the-datalist-element">&lt;datalist&gt;</a>
 * element.
 *
 * <p>
 * The datalist element represents a set of option elements that represent predefined options for other
 * controls. It is used to provide a list of suggested values for input elements, allowing users to
 * either select from the predefined options or enter their own custom value. The datalist element
 * is typically associated with an input element through the list attribute, and the options within
 * the datalist provide autocomplete suggestions.
 *
 * <h5 class='section'>Examples:</h5>
 * <p class='bcode w800'>
 * 	<jk>import static</jk> org.apache.juneau.bean.html5.HtmlBuilder.*;
 *
 * 	<jc>// Simple datalist</jc>
 * 	Datalist <jv>simple</jv> = <jsm>datalist</jsm>(<js>"colors"</js>,
 * 		<jsm>option</jsm>(<js>"red"</js>, <js>"Red"</js>),
 * 		<jsm>option</jsm>(<js>"green"</js>, <js>"Green"</js>),
 * 		<jsm>option</jsm>(<js>"blue"</js>, <js>"Blue"</js>)
 * 	);
 *
 * 	<jc>// Datalist with styling</jc>
 * 	Datalist <jv>styled</jv> = <jsm>datalist</jsm>(<js>"countries"</js>,
 * 		<jsm>option</jsm>(<js>"us"</js>, <js>"United States"</js>),
 * 		<jsm>option</jsm>(<js>"ca"</js>, <js>"Canada"</js>),
 * 		<jsm>option</jsm>(<js>"mx"</js>, <js>"Mexico"</js>)
 * 	)._class(<js>"country-list"</js>);
 *
 * 	<jc>// Datalist with multiple options</jc>
 * 	Datalist <jv>multiple</jv> = <jsm>datalist</jsm>(<js>"fruits"</js>,
 * 		<jsm>option</jsm>(<js>"apple"</js>, <js>"Apple"</js>),
 * 		<jsm>option</jsm>(<js>"banana"</js>, <js>"Banana"</js>),
 * 		<jsm>option</jsm>(<js>"cherry"</js>, <js>"Cherry"</js>),
 * 		<jsm>option</jsm>(<js>"date"</js>, <js>"Date"</js>),
 * 		<jsm>option</jsm>(<js>"elderberry"</js>, <js>"Elderberry"</js>)
 * 	);
 *
 * 	<jc>// Datalist with complex options</jc>
 * 	Datalist <jv>complex</jv> = <jsm>datalist</jsm>(<js>"products"</js>,
 * 		<jsm>option</jsm>(<js>"laptop-001"</js>, <js>"Laptop Pro 15\" - $1,299"</js>),
 * 		<jsm>option</jsm>(<js>"laptop-002"</js>, <js>"Laptop Air 13\" - $999"</js>),
 * 		<jsm>option</jsm>(<js>"tablet-001"</js>, <js>"Tablet 10\" - $499"</js>)
 * 	);
 *
 * 	<jc>// Datalist with ID</jc>
 * 	Datalist <jv>withId</jv> = <jsm>datalist</jsm>(<js>"cities"</js>,
 * 		<jsm>option</jsm>(<js>"new-york"</js>, <js>"New York"</js>),
 * 		<jsm>option</jsm>(<js>"los-angeles"</js>, <js>"Los Angeles"</js>),
 * 		<jsm>option</jsm>(<js>"chicago"</js>, <js>"Chicago"</js>)
 * 	);
 *
 * 	<jc>// Datalist with styling</jc>
 * 	Datalist <jv>styled2</jv> = <jsm>datalist</jsm>(<js>"sizes"</js>,
 * 		<jsm>option</jsm>(<js>"xs"</js>, <js>"Extra Small"</js>),
 * 		<jsm>option</jsm>(<js>"s"</js>, <js>"Small"</js>),
 * 		<jsm>option</jsm>(<js>"m"</js>, <js>"Medium"</js>),
 * 		<jsm>option</jsm>(<js>"l"</js>, <js>"Large"</js>),
 * 		<jsm>option</jsm>(<js>"xl"</js>, <js>"Extra Large"</js>)
 * 	).style(<js>"display: none;"</js>);
 * </p>
 *
 * <p>
 * The following convenience methods are provided for constructing instances of this bean:
 * <ul class='javatree'>
 * 	<li class='jc'>{@link HtmlBuilder}
 * 	<ul class='javatree'>
 * 		<li class='jm'>{@link HtmlBuilder#datalist() datalist()}
 * 		<li class='jm'>{@link HtmlBuilder#datalist(String, Option...) datalist(String, Option...)}
 * 	</ul>
 * </ul>
 * </p>
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/JuneauBeanHtml5">juneau-bean-html5</a>
 * </ul>
 */
@Bean(typeName = "datalist")
public class Datalist extends HtmlElementContainer {

	/**
	 * Creates an empty {@link Datalist} element.
	 */
	public Datalist() {}

	/**
	 * Creates a {@link Datalist} element with the specified {@link Datalist#id(String)} attribute and child nodes.
	 *
	 * @param id The {@link Datalist#id(String)} attribute.
	 * @param children The child nodes.
	 */
	public Datalist(String id, Option...children) {
		id(id).children((Object[])children);
	}

	@Override /* Overridden from HtmlElement */
	public Datalist _class(String value) { // NOSONAR - Intentional naming.
		super._class(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist accesskey(String value) {
		super.accesskey(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist attr(String key, Object val) {
		super.attr(key, val);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist attrUri(String key, Object val) {
		super.attrUri(key, val);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Datalist child(Object value) {
		super.child(value);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Datalist children(Object...value) {
		super.children(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist contenteditable(Object value) {
		super.contenteditable(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist dir(String value) {
		super.dir(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist hidden(Object value) {
		super.hidden(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist id(String value) {
		super.id(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist lang(String value) {
		super.lang(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onabort(String value) {
		super.onabort(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onblur(String value) {
		super.onblur(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oncancel(String value) {
		super.oncancel(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oncanplay(String value) {
		super.oncanplay(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oncanplaythrough(String value) {
		super.oncanplaythrough(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onchange(String value) {
		super.onchange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onclick(String value) {
		super.onclick(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oncuechange(String value) {
		super.oncuechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist ondblclick(String value) {
		super.ondblclick(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist ondurationchange(String value) {
		super.ondurationchange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onemptied(String value) {
		super.onemptied(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onended(String value) {
		super.onended(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onerror(String value) {
		super.onerror(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onfocus(String value) {
		super.onfocus(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oninput(String value) {
		super.oninput(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist oninvalid(String value) {
		super.oninvalid(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onkeydown(String value) {
		super.onkeydown(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onkeypress(String value) {
		super.onkeypress(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onkeyup(String value) {
		super.onkeyup(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onload(String value) {
		super.onload(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onloadeddata(String value) {
		super.onloadeddata(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onloadedmetadata(String value) {
		super.onloadedmetadata(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onloadstart(String value) {
		super.onloadstart(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmousedown(String value) {
		super.onmousedown(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmouseenter(String value) {
		super.onmouseenter(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmouseleave(String value) {
		super.onmouseleave(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmousemove(String value) {
		super.onmousemove(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmouseout(String value) {
		super.onmouseout(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmouseover(String value) {
		super.onmouseover(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmouseup(String value) {
		super.onmouseup(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onmousewheel(String value) {
		super.onmousewheel(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onpause(String value) {
		super.onpause(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onplay(String value) {
		super.onplay(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onplaying(String value) {
		super.onplaying(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onprogress(String value) {
		super.onprogress(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onratechange(String value) {
		super.onratechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onreset(String value) {
		super.onreset(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onresize(String value) {
		super.onresize(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onscroll(String value) {
		super.onscroll(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onseeked(String value) {
		super.onseeked(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onseeking(String value) {
		super.onseeking(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onselect(String value) {
		super.onselect(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onshow(String value) {
		super.onshow(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onstalled(String value) {
		super.onstalled(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onsubmit(String value) {
		super.onsubmit(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onsuspend(String value) {
		super.onsuspend(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist ontimeupdate(String value) {
		super.ontimeupdate(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist ontoggle(String value) {
		super.ontoggle(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onvolumechange(String value) {
		super.onvolumechange(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist onwaiting(String value) {
		super.onwaiting(value);
		return this;
	}

	@Override /* Overridden from HtmlElementContainer */
	public Datalist setChildren(List<Object> children) {
		super.setChildren(children);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist spellcheck(Object value) {
		super.spellcheck(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist style(String value) {
		super.style(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist tabindex(Object value) {
		super.tabindex(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist title(String value) {
		super.title(value);
		return this;
	}

	@Override /* Overridden from HtmlElement */
	public Datalist translate(Object value) {
		super.translate(value);
		return this;
	}
}