/* maketex.c: this program emulates the MakeTeX* shell scripts from web2c.

Copyright (C) 1997 Fabrice POPINEAU.
Adapted to MS-DOS/DJGPP by Eli Zaretskii <eliz@is.elta.co.il>.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Maximum number of lines per usage message. */
#define MAX_LINES 12

#define SEPARATORS " \t\n\""
#define IS_EMPTY(s) ((!s) || !(*(s)))

#include <direct.h>
#include <process.h>
#include <signal.h>

#include <kpathsea/config.h>
#include <kpathsea/absolute.h>
#include <kpathsea/c-ctype.h>
#include <kpathsea/c-fopen.h>
#include <kpathsea/c-stat.h>
#include <kpathsea/c-std.h>
#include <kpathsea/c-memstr.h>
#include <kpathsea/c-pathch.h>
#include <kpathsea/c-pathmx.h>
#include <kpathsea/c-vararg.h>
#include <kpathsea/concatn.h>
#include <kpathsea/db.h>
#include <kpathsea/expand.h>
#include <kpathsea/getopt.h>
#include <kpathsea/line.h>
#include <kpathsea/pathsearch.h>
#include <kpathsea/proginit.h>
#include <kpathsea/tex-file.h>
#include <kpathsea/tex-glyph.h>
#include <kpathsea/variable.h>

#include "stackenv.h"

/* gsftopk defines KPSE_LAST_DEBUG+2, so avoid clashes */
#define MAKETEX_DEBUG (KPSE_LAST_DEBUG + 3)
#define MAKETEX_FINE_DEBUG (KPSE_LAST_DEBUG + 4)

#if 0
#define fopen(f, m) (fprintf(stderr, "opening %s\n", f), fopen(f, m))
#define fclose(f) (fprintf(stderr, "closing %x\n", f), fclose(f))
#endif

#define LAST_CHAR(s) (*((s)+strlen(s)-1))

extern string maketex_version_string;
extern DllImport char* kpathsea_version_string;

/* Variables for MakeTeX* scripts */
static int program_number = -1;
string progname;
static string texmf;
static string texmfmain;
static string texmfcnf_dir;
static char ls_R_magic[] = "% ls-R -- maintained by MakeTeXls-R; do not change this line.";
static char tfmname[PATH_MAX];
static char pkname[PATH_MAX];
static char mt_pkdestdir[PATH_MAX];
static char mt_tfmdestdir[PATH_MAX];
char tmpdir[PATH_MAX]; /* Working temporary directory */
string output = "astdout";
FILE *fout;
static boolean downcase_names;

/* Variables for MakeTeX.site */
static int dpi;
static string mt_bdpi;
static string ps_to_pk;
static string dcfontdir;
static string ecfontdir;
static string fcfontdir;
static string tcfontdir;
static string tsfontdir;
static string sauterfontdir;
static string mt_features;
boolean mt_dosnames = FALSE;
boolean mt_nomode = FALSE;
boolean mt_strip = FALSE;
boolean mt_varfonts = FALSE;

/* Variables for MakeTeXnames.cnf */
static string mt_dir_perms;
static string mt_pkname;
static string mt_namepart;
static string mt_destroot;
static string mt_mode;

static string mt_file_perms;
static mode_t file_perms, dir_perms;

static struct alist {
  string key;
  string *var;
  string val;
} vardef[] = {
  {"MODE", &mt_mode, "ljfour"},
  {"BDPI", &mt_bdpi, "600"},
  {"MT_DIR_PERMS", &mt_dir_perms, "777"},
  {"MT_FILE_PERMS", &mt_file_perms, "666"},
  {"ps_to_pk", &ps_to_pk, "gsftopk"},
#ifdef ENHANCED_MODE
  {"dcfontdir", &dcfontdir, "$TEXMF/fonts/source/jknappen/dc"},
  {"ecfontdir", &ecfontdir, "$TEXMF/fonts/source/jknappen/ec"},
  {"fcfontdir", &fcfontdir, "$TEXMF/fonts/source/jknappen/fc"},
  {"tcfontdir", &tcfontdir, "$TEXMF/fonts/source/jknappen/tc"},
  {"tsfontdir", &tsfontdir, "$TEXMF/fonts/source/jknappen/ts"},
  {"sauterfontdir", &sauterfontdir, "$TEXMF/fonts/source/public/sauter"},
#else /* !ENHANCED_MODE */
  {"dcfontdir", &dcfontdir, "$TEXMFMAIN/fonts/source/jknappen/dc"},
  {"ecfontdir", &ecfontdir, "$TEXMFMAIN/fonts/source/jknappen/ec"},
  {"fcfontdir", &fcfontdir, "$TEXMFMAIN/fonts/source/jknappen/fc"},
  {"tcfontdir", &tcfontdir, "$TEXMFMAIN/fonts/source/jknappen/tc"},
  {"tsfontdir", &tsfontdir, "$TEXMFMAIN/fonts/source/jknappen/ts"},
  {"sauterfontdir", &sauterfontdir, "$TEXMFMAIN/fonts/source/public/sauter"},
#endif /* !ENHANCED_MODE */
#ifdef MSDOS
  {"MT_FEATURES", &mt_features, "dosnames"},
#else
  {"MT_FEATURES", &mt_features, ""},
#endif
  {NULL, NULL, NULL}
};

/* Type of function to execute */
typedef void (*execfn)(string, struct stat*);

/* Test whether getopt found an option ``A''.
   Assumes the option index is in the variable `option_index', and the
   option table in a variable `long_options'.  */
#define ARGUMENT_IS(a) STREQ (long_options[option_index].name, a)

static struct option long_options [] = {
  { "debug",               1, 0, 0},
  { "help",                0, 0, 0},
  { "version",             0, 0, 0},
  {0, 0, 0, 0}
};

/*
  This array describes all the programs that are taken into account.
  Each program is characterized by :
  name, args required, args max, usage_msg  */

void maketexpk(int, char**);
void maketextfm(int, char**);
void maketextex(int, char**);
void maketexmf(int, char**);
void maketexlsr(int, char**);
void maketexupdate(int, char**);
void maketexmkdir(int, char**);
void maketexrmdir(int, char**);
void maketexnames(int, char**);

void rec_rmdir(string);
void do_rmdir(string);
void do_makedir(string);
void delmulslash(string );
string path_in_pathes();

/* Pointer to functions that accept argc and argv[] */
typedef void (*fp)(int, string*);

/* Description of functionality : long name, short name, min args,
   max args and pointer to a function to execute. */
static struct program_description {
  string name;
  string truncated;	/* DOS truncates to 8+3 */
  string altname;
  int arg_min;
  int arg_max;
  fp prog;
} makedesc [] = {
  {"MakeTeXPK",     "MakeTeXP",     "makepk",    4, 6, maketexpk },
  {"MakeTeXTFM",    "MakeTeXT",     "maketfm",   2, 3, maketextfm },
  {"MakeTeXTeX",    "",             "maketex",   4, 6, maketextex },
  {"MakeTeXMF",     "MakeTeXM",     "makemf",    2, 2, maketexmf },
  {"MakeTeXls-R",   "MakeTeXl",     "makelsr",   1, 2, maketexlsr },
  {"MakeTeXupdate", "MakeTeXU",     "makeupd",   3, 3, maketexupdate },
  {"MakeTeX-mkdir", "MakeTeX-",     "makemd",    2, 255, maketexmkdir },
  {"MakeTeXrmdir",  "MakeTeXr",     "makerd",    2, 255, maketexrmdir },
  {"MakeTeXnames",  "MakeTeXn",     "makename",  2, 4, maketexnames },
  {0, 0, 0}
};

/*
  First  argument takes progname;
  Other arguments take no arguments.
  */
static string usage_msg[][MAX_LINES] = {
  /* MakeTeXPK */
  {"Usage: %s NAME DPI BDPI MAG [MODE [DESTDIR]].\n",
   "  Try to create a PK file for NAME at resolution DPI,\n",
   "  with an assumed device base resolution of BDPI,\n",
   "  and a Metafont `mag' of MAG.\n",
   "  Use MODE for the Metafont mode if supplied, unless MODE is `default', in\n",
   "   which case guess as usual. (This is so you can specify DESTDIR without MODE.)\n\n",
   "  Use DESTDIR for the root of where to install into, either the absolute\n",
   "   directory name to use (if it starts with a /) or relative to the default\n",
   "   DESTDIR (if not).\n",
   0
  },
  /* MakeTeXTFM */
  {
    "Usage : %s FONT.\n",
    "  Makes a TFM file for FONT, if possible.\n",
    0
  },
  /* MakeTeXTeX */
  {
    0
  },
  /* MakeTeXMF */
  {
    "Usage : %s FONT.\n",
    "  Makes the Metafont source file for FONT, if possible.\n",
    "  For example, 'dcr12' or 'cmr11'.\n",
    0
  },
  /* MakeTeXls-R */
  {
    "Usage : %s [DIR].\n",
    "  Rebuild the ls-R file completely for the default or specified texmf DIR.\n",
    0
  },
  /* MakeTeXupdate */
  {
    "Usage : %s DIR FILE.\n",
    "  Update the ls-R file with an entry for FILE in DIR.\n",
    0
  },
  /* MakeTeXmkdir */
  {
    "Usage : %s DIRS ...\n",
    "  Create each DIR, including any missing leading directories.\n",
    0
  },
  /* MakeTeXrmdir */
  {
    "Usage : %s DIRS ...\n",
    "  Recursively remove each DIR.\n",
    0
  },
  /* MakeTeXnames */
  {
    "Usage : %s NAME [DPI MODE] [DESTDIR].\n",
    "  Output the PK and TFM names for a font NAME.\n",
    0
  }
};

/* concatenation of pathes */
string concat_pathes(const_string p1, const_string p2)
{
  string res;
  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "Concat pathes %s and %s\n", p1, p2);
  }

  assert (p1 && *p1 && p2 && *p2);

  if (IS_DIR_SEP(LAST_CHAR(p1))) {
    if (IS_DIR_SEP(*p2))
      res = concat(p1, p2+1);
    else
      res = concat(p1, p2);
  }
  else 
    if (IS_DIR_SEP(*p2)) {
      res = concat(p1, p2);
    }
    else
      res = concat3(p1, DIR_SEP_STRING, p2);

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "\t=> %s\n", res);
  }

  return res;
}

/* Reading MakeTeX.site and MakeTeXcommon.  */
static string maketex_site_name = "MakeTeX.site";

void read_mtsite()
{
  FILE *f;
  boolean found;
  string line;
  int len;
  struct alist *p;
  string mtsite;
  string t;


  /* Set default values to some variables, then read MakeTeX.site  */
  for(p = vardef; p->key != NULL; p++) {
    /* testing if there is an env var */
    line = concat("$", p->key);
    *(p->var) = kpse_var_expand(line);
    if (*(p->var) == NULL || **(p->var) == '\0')
      *(p->var) = p->val;
    free(line);
  }

  mtsite = concat(texmfcnf_dir, maketex_site_name);
  if (!test_file('r', mtsite)) {
    free(mtsite);
    mtsite = kpse_find_file(maketex_site_name, kpse_cnf_format, true);
  }

  /* and next overriding those variables from MakeTeX.site */
  if (mtsite && (f = fopen(mtsite, "r")) != NULL) {
    if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
      fprintf(stderr, "Reading %s file: %s\n", maketex_site_name, mtsite);
    }
    while ((line  = read_line(f)) != NULL) {
      if (*line == '#' || *line == '%' || isspace(*line)
	  || *line == '\0' || *line == '\n')
	continue;

      /* Accept keywords that are no-ops on MS platforms, and do nothing.  */
      if (strncmp("umask", line, 5) == 0 && line[5] == '=')
	continue;
      found = FALSE;
      for(p = vardef; !found && p->key != NULL; p++) {
	len = strlen(p->key);
	if (strncmp(p->key, line, len) == 0 && line[len] == '=') {
	  *(p->var) = xstrdup(line+len+1);
	  found = TRUE;
	}
      }
      if (!found) 
	fprintf(stderr, "%s: unsupported keyword: %s\n", mtsite, line);
      free(line);
    }
    fclose(f);
  }

  file_perms = strtol(mt_file_perms, &t, 8);
  if (t != mt_file_perms + 3)
    file_perms = 0666;
  dir_perms  = strtol(mt_dir_perms, &t, 8);
  if (t != mt_dir_perms + 3)
    dir_perms = 0777;
  free(mtsite);
}

void output_and_cleanup(int code)
{
  FILE *f;
  string output_name;
  string line;

  if (code == 0) {
    output_name = concat_pathes(tmpdir, output);

    /* output result if any */
    if ((f = fopen(output_name, "r")) != NULL) {
      while((line = read_line(f)) != NULL) {
	fputs(line, stdout);
	free (line);
      }
      fclose(f);
    }

    free(output_name);
  }

  if (test_file('d', tmpdir))
    rec_rmdir(tmpdir);
}

void usage()
{
  int i;
  fprintf(stderr, "%s of %s\n", progname, kpathsea_version_string);
  fprintf(stderr, "MakeTeX version %s\n", maketex_version_string);
  fprintf(stderr,usage_msg[program_number][0], progname );
  fputs("\n", stderr);
  for(i = 1; usage_msg[program_number][i]; ++i)
    fputs(usage_msg[program_number][i], stderr);
}

/* ensure that the path is written with the same DIR_SEP */
void slashify(string path)
{
  while(*path) {
    if (IS_DIR_SEP(*path))
      *path = DIR_SEP;
    path++;
  }
}

static string texmfcnf_dir_name = "$TEXMFCNF_DIR";

int main(int argc, char* argv[])
{
  int i,j;
  int g; /* getopt return code */
  int option_index;
  struct program_description * program = 0;
  string tempenv;
  /* `kpse_init_prog' will put MODE and DPI into environment as values of
     the two variables below.  We need to leave these variables as they are. */
  string mode = getenv("MAKETEX_MODE");
  string bdpi = getenv("MAKETEX_BASE_DPI");
  unsigned idpi = bdpi ? atoi(bdpi) : 600;
  FILE *fnul;
  int newfd[3];
  string tstr;

  if (!progname)
    progname = argv[0];
  kpse_set_progname (progname);

  /* NULL for no fallback font. */
  kpse_init_prog (uppercasify (progname), idpi, mode, NULL);

  for(i = 0; makedesc[i].name; ++i)
    if (FILESTRCASEEQ(program_invocation_short_name, makedesc[i].name)
        || FILESTRCASEEQ(program_invocation_short_name, makedesc[i].truncated)
        || FILESTRCASEEQ(program_invocation_short_name, makedesc[i].altname)) {
      program_number = i;
      progname = makedesc[i].name;
      program = makedesc+i;
      break;
    }

  if (!makedesc[i].name) {
    fprintf(stderr, "This program was incorrectly copied to the name %s\n", 
	    argv[0]);
    return 1;
  }

  for(;;) {
    g = getopt_long_only (argc, argv, "", long_options, &option_index);

    if (g == EOF)
      break;

    if (g == '?')
      return 1;  /* Unknown option.  */

    /* assert (g == 0); */ /* We have no short option names.  */
    
    if (ARGUMENT_IS ("debug")) {
      kpathsea_debug |= atoi (optarg);
    }
    else if (ARGUMENT_IS ("help")) {
      usage();
      return 0;
    }
    else if (ARGUMENT_IS ("version")) {
      fprintf(stderr, "%s of %s.\n", progname, kpathsea_version_string);
      return 0;
    }

  }
#if 0
  fprintf(stderr, "Before Arg count = %d\n", argc);
  for (i = 0; i < argc; i++)
    fprintf(stderr, "argv[%d] = %s\n", i, argv[i]);
#endif
  /* shifting options from argv[] list */
  for (i = j = 1; i < argc; i++) {
    if (*(argv[i]) != '-') {
      argv[j] = argv[i];
      j++;
    }
  }
  argc = j;
#if 0
  fprintf(stderr, "After Arg count = %d\n", argc);
  for (i = 0; i < argc; i++)
    fprintf(stderr, "argv[%d] = %s\n", i, argv[i]);
#endif
  if ((argc < program->arg_min)
      || (argc  > program->arg_max)) {
    fprintf(stderr, 
	    "Bad number of arguments. Use %s --help if you need it.\n", 
	    progname);
    return 1;
  }

  /* texmf may contain multiple pathes ! */
  texmf = kpse_path_expand("$TEXMF");

#ifdef ENHANCED_MODE
    /* No error if texmfmain is not found : it is a fallback value,
       used only if nothing else is found */ 
  texmfmain = kpse_var_expand("$TEXMFMAIN");

  /* This should be where texmf.cnf is found */
  texmfcnf_dir = dirname(kpse_find_file("texmf.cnf", kpse_cnf_format, true));

  if (!texmfmain || !*texmfmain) 
    texmfmain = path_in_pathes(texmfcnf_dir, texmf);

#else /* !ENHANCED_MODE */

  /* All MakeTeX... scripts do this:

     : ${TEXMFMAIN=`kpsewhich -expand-var='$TEXMFMAIN'`}
     if test -z "$TEXMFMAIN"; then
       echo "$0: No \$TEXMFMAIN; set the environment variable or in texmf.cnf."
       exit 1
     fi

     Let's do it also.  */
  texmfmain = kpse_var_expand("$TEXMFMAIN");
  if (!texmfmain || !*texmfmain) {
    fprintf(stderr,
	    "%s: No $TEXMFMAIN; set the environment variable or in texmf.cnf.",
	    progname);
    mt_exit(1);
  }

  /* 
     : ${TEXMFCNF_DIR=`kpsewhich -expand-var='$TEXMFCNF_DIR'`}
     test -n "$TEXMFCNF_DIR" || TEXMFCNF_DIR=$TEXMFMAIN/web2c     */
  texmfcnf_dir = kpse_var_expand(texmfcnf_dir_name);
  if (!texmfcnf_dir || !*texmfcnf_dir) {
    texmfcnf_dir = concat(texmfmain, "/web2c");
    xputenv(texmfcnf_dir_name + 1, texmfcnf_dir);
  }
#endif /* !ENHANCED_MODE */

  /* . $TEXMFCNF_DIR/MakeTeXcommon  */
  tempenv = getenv("TMPDIR");
  if (!tempenv)
    tempenv = getenv("TEMP");
  if (!tempenv)
    tempenv = getenv("TMP");
#ifdef DOSISH
  if (!tempenv)
    tempenv = "c:/tmp";	/* "/tmp" is not good if we are on a CD-ROM */
#else
  if (!tempenv)
    tempenv = "/tmp";
#endif
  sprintf(tmpdir, "%s/mtXXXXXX", tempenv);
  mktemp(tmpdir);
  dir_perms = 0755;	/* MT_DIR_PERMS was not set yet */
  do_makedir(tmpdir);
  xputenv("TEMPDIR", tmpdir);
  xputenv("STDOUT", tstr = concat_pathes(tmpdir, output));
  free(tstr);

  xputenv("KPSE_DOT", tstr = xgetcwd());
  free(tstr);

  pushd(tmpdir);
  
  if ((fout = fopen(output, "w")) == NULL) {
    perror(output);
    mt_exit(1);
  }

  /* Catch signals, so we clean up if the child is interrupted.  */
#ifdef _WIN32
  SetConsoleCtrlHandler((PHANDLER_ROUTINE)HandlerRoutine, TRUE);
#else
  signal (SIGINT, sigint_handler);
#endif
  read_mtsite();

#ifdef _WIN32
  fnul = fopen("nul", "r");	/* fopen /dev/null */
#else
  fnul = fopen("/dev/null", "r");
#endif
  
  newfd[0] = fileno(fnul);
  newfd[1] = 2;
  newfd[2] = 2;
  
  push_fd(newfd);
  program->prog(argc, argv);
  
  fclose(fnul);
  fclose(fout);

  mt_exit(0);
}

/*
  Recursive walk through the directory tree. Depth-first order. 
*/
void recurse_dir(string path, execfn before, execfn after)
{
    /* In depth traversal of the subdir tree */
#if defined(_WIN32)
  WIN32_FIND_DATA find_file_data;
  HANDLE hnd;
  int index;
#else
  DIR *dp;
  struct dirent *ep;
#endif
  struct stat stat_buf;	/* We have to have one local because
			   of after */

  int path_len = strlen(path);

  /* current node */
  if (stat(path, &stat_buf))
    perror(path);

  /* execute before for the current node */
  if (before)
    (*before)(path, &stat_buf);

  /* if it is a directory, recurse through all sons */
  if ((stat_buf.st_mode & S_IFMT) == S_IFDIR) {
#if defined(_WIN32)
    strcat(path, "/*");
#if 0
    fprintf(stderr, "Opening hnd on %s\n", path);
#endif
    hnd = FindFirstFile(path, &find_file_data);
    while (hnd != INVALID_HANDLE_VALUE && 
	   FindNextFile(hnd, &find_file_data) != FALSE) { 
      if(!strcmp(find_file_data.cFileName, ".")
	 || !strcmp(find_file_data.cFileName, "..")) 
	continue;
      path[path_len+1] = '\0';
      strcat(path, find_file_data.cFileName);
      recurse_dir(path, before, after);
    }
    path[path_len] = '\0';
#if 0
    fprintf(stderr, "Closing hnd on %s\n", path);
#endif
    FindClose(hnd);
#else
    if ((dp = opendir(path))) {
      while ((ep = readdir(dp)))
	if (strcmp(ep->d_name, ".") &&
	    strcmp(ep->d_name, "..")) {
	  path[path_len] = '/';
	  strcpy(path+path_len+1, ep->d_name);
	  recurse_dir(path, before, after);
	}
      path[path_len] = '\0';
      closedir(dp);
    }
    else
      perror(path);
#endif
  }
  /* execute after for the current node */
  if (after)
    (*after)(path, &stat_buf);
}

/* remove something which already exists */
void remove_path(string path, struct stat* st)
{
  int e = errno;
#if 0
  fprintf(stderr, "Removing %s.\n", path);
#endif
  if ((st->st_mode & S_IFMT) == S_IFDIR) {
    errno = 0;
    if (rmdir(path) == -1 && errno == EBUSY) {
      /* `rmdir' will fail (at least on MS-DOS) if PATH is the current
	 working directory on another drive, or the parent of the
	 working directory.  To work around that, we chdir to the root
	 of that drive, then try to rmdir again.  */
      char drv_root[4], *p = drv_root;

      if (path[0] && path[1] == ':') {
	*p++ = path[0];
	*p++ = path[1];
      }
      *p++ = '/';
      *p = '\0';
      pushd(drv_root);
      if (rmdir(path) == 0) {
	errno = e;
	popd();
	return;
      }
    }
    else if (errno)
      perror(path);
    else
      errno = e;
  }
  else {
    if (unlink(path) == -1)
      perror(path);
  }
}

/* Creates path, and all intermediate directories if necessary. */

void do_makedir(string path)
{
  struct stat stat_buf;
  string p = path;

  /* We can't authorize multiple slashes here */
  delmulslash(path);

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "Creating directory: %s\n", path);
  }

  if (stat(path, &stat_buf) == 0 &&
      (stat_buf.st_mode & S_IFMT) == S_IFDIR)
    return;			/* The path already exists */

  /* Finding the [drive letter][:] */
  if (IS_DEVICE_SEP(*(p+1)))
    p +=2;

  if (IS_DIR_SEP(*p))
    p++;

  for ( ; *p ; p++) {
    if (IS_DIR_SEP(*p)) {
      *p = '\0';
      if ((mkdir(path, dir_perms) == -1)
	  && (errno != EEXIST)) { /* We do it for each path elt, even if it
				     already exists */
	perror(path);
	fprintf(stderr, "%s: MakeTeX-mkdir %s failed.\n", progname, path);
	mt_exit(1);
      }
      chmod(path, dir_perms);	/* probably a no-op, but why not? */
      *p = DIR_SEP;
    }
  }
  /* and for the whole path */
  if (mkdir(path, dir_perms) == -1) {
    perror(path);
    fprintf(stderr, "%s: MakeTeX-mkdir %s failed.\n", progname, path);
    mt_exit(1);
  }
  chmod(path, dir_perms);
}

/* Output the string in lower case or without modification according
   to downcase_names. */
void output_name(string name, FILE *f)
{
  string p;
  if (downcase_names) {
    for(p = name; *p; p++)
      *p = TOLOWER(*p);
  }
  fputs(name, f);
}

/* Updates ls-R db for directory pointed to by path. */
void print_path(string path, struct stat* st)
{
  if ((st->st_mode & S_IFMT) == S_IFDIR) {
#ifdef _WIN32
    WIN32_FIND_DATA ffd;
    HANDLE hnd;
    int index = strlen(path);
#else
    DIR *dp;
    struct dirent *ep;
#endif

    putchar('\n');	/* extra blank line before new directory */
    output_name(path, stdout);
    if (path[0] == '.' && path[1] == '\0')
      putchar('/');
    putchar(':'); putchar('\n');

    /* Note that we actually read each directory twice: once
       here, to print all its entries, and once in `recurse_dir',
       to recurse into its subdirectories.  This is so that
       `ls-R' file is written in breadth-first manner, like
       `ls -R' does in the original script.  Caching the entries
       from the first read would make this code run faster, but
       since it is faster than the script already, why bother?  */

#ifdef _WIN32
    strcat(path, "/*");	  /* ok since the caller says path[PATH_MAX] */
    hnd = FindFirstFile(path, &ffd);
    while (hnd && FindNextFile(hnd, &ffd)) { 
      if (ffd.cFileName[0] == '.'
	  && ((ffd.cFileName[1] == '\0')
	      || (ffd.cFileName[1] == '.' && ffd.cFileName[2] == '\0')))
	continue;
      output_name(ffd.cFileName, stdout);
      putchar('\n');/* We do want a \n */
    }
    path[index] = '\0';
    FindClose(hnd);
#else  /* not _WIN32 */
    if ((dp = opendir(path))) {
      while ((ep = readdir(dp)))
	if (strcmp(ep->d_name, ".") &&
	    strcmp(ep->d_name, "..")) {
	  output_name(ep->d_name, stdout);
	  putchar('\n');
	}
      closedir(dp);
    }
    else
      perror(path);
#endif /* not _WIN32 */

  }
}

/* return true if the fs on which path lies is case sensitive.
   Insufficient. Some people may have copied the complete tree in
   upper case from a case insensitive fs.
   Second test : look at the texmf name. If GetShortName !=
   GetLongName it is ok. 
*/
boolean is_casesensitive(string path)
{
  boolean retval = TRUE;	/* default for non-DOSISH systems */

#ifdef MONOCASE_FILENAMES
  char name[PATH_MAX];
#ifdef _WIN32
  char volname[PATH_MAX];
  DWORD sernum;
  DWORD maxcomplen;
#endif
#ifdef __DJGPP__
  int   maxnamelen;
#endif
  DWORD maxpathlen;
  DWORD fsflags;
  char fsname[PATH_MAX];
  
  /* How should the db be output ? Preserve case or lower case ? 
     First, determine the root volume : if path has the drive, ok. If
     not, get current drive. */

  if (IS_DEVICE_SEP(path[1])) {	/* the shortest path is "." and has 2 chars */
    name[0] =  path[0];
  }
  else
    name[0] = _getdrive() + 'A' - 1;

  strcpy(name+1, ":\\");

#ifdef __DJGPP__
  fsflags = _get_volume_info(name, &maxnamelen, &maxpathlen, fsname);
#endif
#ifdef _WIN32
  if (GetVolumeInformation(name, volname, 256, &sernum, &maxcomplen,
			   &fsflags, fsname, 256) == FALSE) {
    fprintf(stderr, "Error getting volume info on %s.\n", name);
    mt_exit(1);
  }
#endif

  retval = (fsflags & FS_CASE_IS_PRESERVED) != 0;

#endif /* MONOCASE_FILENAMES */

  return retval;
}

void do_maketexlsr(string path)
{
  static char name[PATH_MAX];
#ifdef __DJGPP__
  /* `stat' can be very expensive on MS-DOS, and `recurse_dir'
     uses it a lot. Make it work as fast as we possibly can afford.  */
  unsigned short save_stat_bits = _djstat_flags;
  _djstat_flags = (_STAT_WRITEBIT   | _STAT_ROOT_TIME | _STAT_DIRSIZE |
		   _STAT_EXEC_MAGIC | _STAT_EXEC_EXT);
#endif

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "Building ls-R database in: %s\n", path);
  }

  downcase_names = ! is_casesensitive(path);

  strcpy(name,path);
  recurse_dir(name, print_path, NULL);
#ifdef __DJGPP__
  _djstat_flags = save_stat_bits;
#endif
}

static string ls_R = "ls-R";
static string texmfls_r_name = "$TEXMFLS_R";

void maketexlsr(int argc, char **argv)
{
  /* We have to take multiple root trees into account. */

  string lsrname, lsrdir, elt;
  char tmpname[PATH_MAX];
  FILE* lsrfile;
  int newfd[3];

#ifdef ENHANCED_MODE
  if (argc == 1)
    lsrdir = xstrdup(texmf);
  else {
    lsrdir = kpse_path_expand(argv[1]);
  }
  for (elt = kpse_path_element(lsrdir); elt;
       elt = kpse_path_element(NULL))
#else
    if (argc > 1) {
      lsrdir = xstrdup(argv[1]);
      xputenv(texmfls_r_name + 1, lsrdir);
    }
    else {
      /* If no argument, we must use the default texmf tree.*/
      lsrdir = kpse_var_expand(texmfls_r_name);
      
      if (!lsrdir || !*lsrdir) {
      lsrdir = texmfmain;
      xputenv(texmfls_r_name + 1, lsrdir);
      }
    }
  elt = lsrdir;
#endif
  {
    if (!test_file('d', elt)) {
      fprintf(stderr, "%s: %s is not a directory.\n", progname, elt);
      mt_exit(1);
    }

    lsrname = concat_pathes(elt, "/ls-R");

    sprintf(tmpname, "%s/lsXXXXXX", elt);	/* temporary ls-R file */
    if (mktemp(tmpname) == NULL) {
      perror("lsXXXXXX");
      mt_exit(1);
    }
    if ((lsrfile = fopen(tmpname, "w")) == NULL) {
      perror(lsrname);
      mt_exit(1);
    }

    fprintf(stderr, "MakeTeXls-R: Updating %s...\n", lsrname);
  
    newfd[0] = 0;
    newfd[1] = fileno(lsrfile);
    newfd[2] = 2;
    push_fd(newfd);
    
    puts(ls_R_magic);
    fputs("/dev/null", stdout);	/* like `ls -R' does */
    putchar('\n');
    
    pushd(elt);
    do_maketexlsr(".");
    
    popd();			/* back to tmpdir */
    /* Restore original stdout */
    pop_fd();

    fflush(stdout);
    fclose(lsrfile);
    
    if (mvfile(tmpname, lsrname) == FALSE) {
      fprintf(stderr, "%s: can't copy %s to %s.\n",
	      progname, tmpname, lsrname);
      mt_exit(1);
    }
    free(lsrname);
  }
  fprintf(stderr, "MakeTeXls-R: Done.\n");
  free(lsrdir);
}

/* Find namepart for font name in the TDS structure. */
string findmap(string name)
{
  FILE *f;
  string line;
  string token;
  string filename;
  string namepart = NULL;
  string sourcedir = "";
  string typeface = "";
  string fontname = kpse_var_expand("$FONTNAME");
  string typefacedir = NULL;

#ifdef ENHANCED_MODE
  if (fontname != NULL && *fontname != '\0')
    filename = concat_pathes(fontname, "/special.map");
  else {
    filename = kpse_find_file("special.map", kpse_fontmap_format, true);
    if (!filename) {
      filename = concat_pathes(texmfmain, "/fontname");
      if (!test_file('r', filename))
	filename = NULL;
    }
  }
#else /* !ENHANCED_MODE */
  if (fontname == NULL || *fontname == '\0')
    fontname = concat_pathes(texmfmain, "/fontname");

  filename = concat_pathes(fontname, "/special.map");
  if (!test_file('r', filename))
    /* Last resort: search the default paths.  I don't think
       the original MakeTeXnames script does this, somehow.  */
    filename = kpse_find_file("special.map", kpse_fontmap_format, true);
#endif /* !ENHANCED_MODE */
  if (filename == NULL)
    return NULL;
  f = fopen(filename, FOPEN_R_MODE);
  if (f == NULL)
    return NULL;

  /* FIXME : there is a seg fault if the font is not found here */
  /* Is it ok ? 03/02/97 ;.. */
  while((line = read_line(f)) != NULL) {
    token = strtok(line, SEPARATORS);
    if (!token || !*token)
      continue;
#if 0
    fprintf(stderr, "%s %s %d %c\n", token, name, strlen(token), name[strlen(name)-1]);
#endif
    if (FILESTRCASEEQ(token, name)
	|| (FILESTRNCASEEQ(token, name, strlen(token))
	    && isdigit(name[strlen(name)-1])
	    && !isdigit(token[strlen(token)-1]))) {
      sourcedir = strtok(NULL, SEPARATORS);
      typeface = strtok(NULL, SEPARATORS);
      namepart = concat_pathes(sourcedir, typeface);
      free(line);
      break;
    }
    free(line);
  }
  if (!namepart || !*namepart) {
    free(filename);
    filename = concat_pathes(fontname, "/supplier.map");
    if (!test_file('r', filename))
      filename = kpse_find_file("supplier.map", kpse_fontmap_format, true);
    f = filename ? freopen(filename, FOPEN_R_MODE, f) : NULL;
    if (f == NULL)
      return NULL;

    while((line = read_line(f)) != NULL) {
      token = strtok(line, SEPARATORS);
      if (FILECHARCASEEQ(*token, *name) && token[1] == '\0') {
	sourcedir = xstrdup(strtok(NULL, SEPARATORS));
	free(line);
	break;
      }
      free(line);
    }
    free(filename);
    if (sourcedir && *sourcedir && strlen(name) > 2) {
      filename = concat_pathes(fontname, "/typeface.map");
      if (!test_file('r', filename))
	filename = kpse_find_file("typeface.map", kpse_fontmap_format, true);
      f = filename ? freopen(filename, FOPEN_R_MODE, f) : NULL;
      if (f == NULL)
	return NULL;
      while((line = read_line(f)) != NULL) {
	token = strtok(line, SEPARATORS);
	if ((strlen(token) == 2)
	    && (name[1] == token[0]) && (name[2] == token[1])) {
	  typefacedir = xstrdup(strtok(NULL, SEPARATORS));
	  free(line);
	  break;
	}
	free(line);
      }
      if (typefacedir && *typefacedir) {
	namepart = concat_pathes(sourcedir, typefacedir);
	free(sourcedir);
	free(typefacedir);
      }
      else
	fprintf(stderr, "%s: Could not map typeface abbreviation %c%c.\n",
		progname, name[1], name[2]);
    }
    else
      fprintf(stderr, "%s: Could not map source abbreviation %c.\n",
	      progname, name[0]);
  }
  free(filename);
  fclose(f);

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "Font %s has %s as namepart.\n", name, namepart);
  }

  return namepart;
}

void delmulslash(string path)
{
  string p, q;
#if 0
  fprintf(stderr, "Delmulslash before : %s\n", path);
#endif
  /* Delete any multiple slashes */
  for (p = q = path; *p != '\0'; p++, q++) {
    if (*p == '/') {
      *q = *p;
      q++;
      while (*++p == '/');
    }
    *q = *p;
  }
  *q = '\0';
#if 0
  fprintf(stderr, "Delmulslash after : %s\n", path);
#endif
}  

/* Test if two pathes are equal. Up to the shortest one. */
boolean path_included(const_string p1, const_string p2)
{
  boolean ret = true;

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "Path %s included in %s: ", 
	    p1, p2);
  }

  for( ; *p1 && *p2 && ret; p1++, p2++)
    ret = ret && (IS_DIR_SEP(*p1) ? IS_DIR_SEP(*p2) :
		  TOUPPER(*p1) == TOUPPER(*p2));

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "%s\n", (ret ? "true": "false"));
  }
  
  return ret;
}

/* return the path element in path which can match dir else NULL */
string path_in_pathes(const_string dir, const_string path)
{
  string ret = NULL;
  string elt;
  string pathcpy = xstrdup(path);
  /* FIXME : why is it necessary to dup it ? */
  string dircpy = xstrdup(dir);
#if 0
  fprintf(stderr, "Path_in_pathes : dir = %s\n", dir);
#endif
  for (elt = kpse_path_element (pathcpy); 
       elt && !ret;
       elt = kpse_path_element (NULL)) {
    /* Not sufficient to have both equal, there should be a DIR_SEP
       on the next char of dir */
#if 0
    fprintf(stderr, "Dir = %s; elt = %s\n", dir, elt);
#endif
    if (path_included(elt, dircpy) &&
	(IS_DIR_SEP(*(dir+strlen(elt)))
	 || (strlen(dircpy) == strlen(elt))))
      ret = xstrdup(elt);
  }

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "Path %s searched for %s => %s.\n", 
	    path, dircpy, ret);
  }
  free(pathcpy);
  free(dircpy);
  return ret;
  
}

#ifdef ENHANCED_MODE

/* Look if .tfm file exists, else if .mf file exists. Will have
   to enhance this for MakeTeXMF. */
string try_to_find_root(const_string name)
{
  string tfm_name = concat(name, ".tfm");
  string mf_name = concat(name, ".mf");
  
  string location;
  string elt, p;
  string ret;

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) { 
    fprintf(stderr, "Try to find root : %s\ntfm = %s, mf = %s\n", name, tfm_name, mf_name);
    fprintf(stderr, "%s : %s\n", 
	    tfm_name, kpse_find_file(tfm_name, kpse_tfm_format, true));
    fprintf(stderr, "%s : %s\n", 
	    mf_name, kpse_find_file(mf_name, kpse_mf_format, true));
  }

  location = kpse_find_file(tfm_name, kpse_tfm_format, true);
  if (location == NULL)
    location = kpse_find_file(mf_name, kpse_mf_format, true);

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "Location : %s\n", location);
  }
  
  if (location && *location) {
    /* Looking for which path_element in texmf is matched by location. */
    ret = path_in_pathes(location, texmf);
    free(location);
  }
  else {
    /* No location found */
    fprintf(stderr, "%s: can't find neither %s nor %s.\nI won't be able to build your file.\n",
	    progname, tfm_name, mf_name);
    mt_exit(1);
  }

  free (tfm_name);
  free (mf_name);
  

  /* We have a texmf tree in which to put something.
     Is it possible ? At first sight, yes if it is not on
     a CD-ROM ... in which case we return $TEXMFMAIN */
  if (ret && NAME_BEGINS_WITH_DEVICE(ret)
      && IS_DIR_SEP(ret[2])) {
    char drv[4];
    int type;
    strncpy(drv, ret, 3);
    drv[3] = '\0';
    if ((type = GetDriveType(drv)) == DRIVE_CDROM
	/*	|| type == 0 The drive type cannot be determined ? */
	|| type == 1) {
      free(ret);
      ret = xstrdup(texmfmain);
    }
#if 0
    fprintf(stderr, "Drive type : %d\n", type);
#endif
  }

  if (!ret) {
    ret = kpse_var_expand("$TEXMFMAIN");
    if (!ret)
      oops("%s: No $TEXMFMAIN; set the environment variable or in texmf.cnf.\n", progname);
  }

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "Root for font %s is: %s\n", name, ret);
  }

  return ret;
}

#endif /* ENHANCED_MODE */

void read_maketexnames_cnf(void)
{
  string t;

  /* Processing of mt_features (roughly equivalent to MakeTeXnames.cnf) */
  t = strtok(mt_features, SEPARATORS);
  while(t != NULL && *t != '\0') {
    if (KPSE_DEBUG_P(MAKETEX_DEBUG))
      fprintf(stderr, "MT_FEATURES : %s\n", t);
    if (strcmp(t, "dosnames") == 0) {
      mt_dosnames = TRUE;
      mt_pkname = "dpi%d/%s.pk";
    }
    else if (strcmp(t, "nomode") == 0) {
      mt_nomode = TRUE;
      mt_mode = "";
    }
    else if (strcmp(t, "strip") == 0) {
      mt_strip = TRUE;
      mt_namepart = "";
    }
    else if (strcmp(t, "varfonts") == 0) {
      /*  A user can override this setting in either direction by setting
	  USE_VARTEXFONTS to 1 or 0.  */
      string use_vartexfonts = kpse_var_value("USE_VARTEXFONTS");
      string vartexfonts;

      if (use_vartexfonts && use_vartexfonts[0] != '0') {
	vartexfonts = kpse_var_expand("$VARTEXFONTS");
	if (!vartexfonts || !*vartexfonts) {
	  fprintf(stderr,
	  "%s: You asked for varfonts in MT_FEATURES, but VARTEXFONTS\n",
	  progname);
	  fprintf(stderr,
	  "%s: is not set as an environment variable or in texmf.cnf, etc.\n",
	  progname);
	}
	else {
	  mt_varfonts = TRUE;
	  mt_destroot = vartexfonts;
	  mt_namepart = "";
	}
      }
    }
    t = strtok(NULL, SEPARATORS);
  }

  /* Note: the feature whereby MakeTeX.site is sourced again after
     reading MakeTeXnames.cnf is NOT supported here.  (People who need
     this will probably use some clever shell techniques there, and we
     are not *really* a shell interpreter, we are just an impostor...)  */
}

void do_maketexnames(string name, string dest, int dpi, string mode)
{
  string mt_tfmpart = "tfm";
  string mt_pkpart = "pk";
  string p;

  mt_pkname = "%s.%dpk";
  mt_pkdestdir[0] = '\0';

  mt_namepart = kpse_var_expand("$MT_NAMEPART");
  if (!mt_namepart || !*mt_namepart) {
    /* Find MT_NAMEPART in *.map.
       Note: MakeTeXnames also allows for $MT_NAMEPART
       to be put into the environment, but we don't.  */
    if ((mt_namepart = findmap(name)) == NULL || *mt_namepart == '\0')
      mt_namepart = "tmp";
  }
  
  if (mode)
    mt_mode = mode;

  /* . $TEXMFCNF_DIR/MakeTeXnames.cnf  */
  read_maketexnames_cnf();

  /* If $TEXMFLS_R is set, generated fonts go to that tree.
     If not, fall back to $TEXMFMAIN.  */
  p = kpse_var_expand(texmfls_r_name);
  if (!p || !*p) {
#ifdef ENHANCED_MODE
    if (mt_varfonts == FALSE)
      p = try_to_find_root(name);
#else  /* !ENHANCED_MODE */
    p = xstrdup(texmfmain);
#endif /* !ENHANCED_MODE */
  }
  mt_destroot = concat_pathes(p, "/fonts");
  free(p);

  if (dest) {
    if (kpse_absolute_p(dest, false))
      strcpy(mt_pkdestdir, dest);
    else
      mt_namepart = dest;
  }

  if (!*mt_pkdestdir)
    sprintf(mt_pkdestdir, "%s/%s/%s/%s",
	    mt_destroot, mt_pkpart, mt_mode, mt_namepart);
  sprintf(mt_tfmdestdir, "%s/%s/%s", mt_destroot, mt_tfmpart, mt_namepart);

  /* eval PKNAME=$MT_PKDESTDIR/$MT_PKNAME  */
  if (mt_dosnames)
    sprintf(pkname, concat3(mt_pkdestdir, "/", mt_pkname), dpi, name);
  else
    sprintf(pkname, concat3(mt_pkdestdir, "/", mt_pkname), name, dpi);

  /* eval TFMNAME=$MT_TFMDESTDIR/$MT_TFMNAME  */
  sprintf(tfmname, "%s/%s.tfm", mt_tfmdestdir, name);

  /* When "dosnames" is in effect, pkdestdir should include "/dpiNNN".  */
  p = dirname(pkname);
  strcpy(mt_pkdestdir, p);
  free(p);
  
  delmulslash(mt_pkdestdir );
  delmulslash(mt_tfmdestdir);
  delmulslash(pkname);
  delmulslash(tfmname);

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MakeTeXnames:\n\tFont: %s\n\tDestination: %s\n\tDpi: %d\n\tMode: %s\n", name, dest ? dest : "(nil)", dpi, mode ? mode : "(nil)");
    fprintf(stderr, "\t=>:\n\tPK dst. dir: %s\n\tTFM dest. dir: %s\n\tPK name: %s\n\tTFM name %s\n", mt_pkdestdir, mt_tfmdestdir, pkname, tfmname);
  }

}

void maketexnames(int argc, char **argv)
{
  string name = argv[1];
  string dest = NULL, mode = NULL;

  argc--; argv++; /* We are eliminating argv[0] */
  if (argc <= 2) {
    dpi = atoi(mt_bdpi);
    if (argc == 2)
      dest = argv[1];
  }
  else if (argc < 5) {
    dpi = atoi(argv[1]);
    mode = argv[2];
    if (argc == 4)
      dest = argv[3];
  }

  do_maketexnames(name, dest, dpi, mode);

  fputs(pkname, fout); fputc(' ', fout);
  fputs(tfmname, fout); fputc(' ', fout);
  fputc('\n', fout);
}

void maketexmkdir(int argc, char **argv)
{
  int i;
  string path;
  for(i = 1; i < argc; i++) {
    /* We have to concat the original wd if argv[i] is not absolute */
    if (kpse_absolute_p(argv[i], false))
      do_makedir(argv[i]);
    else {
      /* original path is in stack_env[0] */
      path = concat_pathes(peek_dir(0), argv[i]);
      do_makedir(path);
      free(path);
    }
  }
      
}

void do_rmdir(string path)
{
  static char name[PATH_MAX];
  strcpy(name,path);

  if (test_file('d', path)) {
    recurse_dir(name, NULL, remove_path);
#if 0
    if (rmdir(path) == -1) perror(path);
#endif
  }
  else
    unlink(path);
}

/* Recursively destructs files & directories from path. */
void rec_rmdir(string path)
{
  string p;
#if 0
  fprintf(stderr, "Path : %s, l = %d, last : %c\n", path, strlen(path), path[strlen(path)-1]);
#endif
  for(p=path+strlen(path)-1; IS_DIR_SEP(*p) && (p > path) ; p--) {
    *p = '\0';
  }
  do_rmdir(path);
}

void maketexrmdir(int argc, char **argv)
{
  int i;

  for(i = 1; i < argc; i++) {
      do_rmdir(argv[i]);
  }
      
}

void do_maketexupdate(string destdir, char *name)
{
  string path, db_file, destdirname, relative_destdir;
  char buf[sizeof ls_R_magic];
  FILE *f;
  string lsrdir = kpse_var_expand(texmfls_r_name);

#ifdef ENHANCED_MODE
 if (!lsrdir || !*lsrdir) {
   lsrdir = path_in_pathes(destdir, texmf);
   if (!lsrdir || !*lsrdir) {
     lsrdir = path_in_pathes(destdir, 
			     kpse_var_expand("$VARTEXFONTS"));
     if (!lsrdir || !*lsrdir) {
       lsrdir = texmfmain;
       if (!lsrdir || !*lsrdir) {
	 fprintf(stderr, 
		 "%s: MakeTeXupdate found that %s isn't a directory in the %s tree.\n", 
		 progname, destdir, texmf);
	 mt_exit(1);
       }
     }
   }
 }
#else  /* !ENHANCED_MODE */
  if (!lsrdir || !*lsrdir) {
    lsrdir = texmfmain;
  }
#endif /* !ENHANCED_MODE */
  xputenv(texmfls_r_name+1, lsrdir);

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MakeTeXupdate adds %s in %s\n", name, destdir);
  }

  slashify(destdir);
  if (!test_file('d', destdir)) {
    fprintf(stderr, "%s: %s is not a directory.\n", progname, destdir);
    mt_exit(1);
  }
  path = concat_pathes(destdir, name);
  if (!test_file('f', path)) {
    fprintf(stderr, "%s: %s is not a file.\n", progname, path);
    mt_exit(1);
  }
  free(path);
  /* We should look for each path_elt of texmf ! */
  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "Destdir = %s\n", destdir);
    fprintf(stderr, "first : %s\n", kpse_var_expand("$TEXMFLS_R"));
    fprintf(stderr, "Destdir = %s\n", destdir);
    fprintf(stderr, "second : %s\n", path_in_pathes(destdir, texmf));
    fprintf(stderr, "Destdir = %s\n", destdir);
    fprintf(stderr, "third : %s\n", path_in_pathes(destdir, kpse_var_expand("$VARTEXFONTS")));
  }

  db_file = concat_pathes(lsrdir, ls_R);
  if (!test_file('f', db_file)) {
    char *arg[2];
    arg[0] = "MakeTeXls-R";
    arg[1] = lsrdir;
    maketexlsr(2, arg);
    if (!test_file('f', db_file)) {
      fprintf(stderr, "%s: %s does not exist.\n", progname, db_file);
      mt_exit(1);
    }
  }
  if ((f = fopen(db_file, "r+")) == NULL) {
    fprintf(stderr, "%s: %s unwritable.\n", progname, db_file);
    mt_exit(1);
  }
  if (!fgets(buf, sizeof buf, f) || strcmp(buf, ls_R_magic)) {
    fprintf(stderr, "%s: %s lacks magic string\n%s.\n", progname, db_file,
	    ls_R_magic);
    mt_exit(1);
  }

  /* does the fs preserve case ? */
  if (!is_casesensitive(destdir)) {
    string p;
    for(p = destdir; *p; p++)
      *p = TOLOWER(*p);
    for(p = name; *p; p++)
      *p = TOLOWER(*p);
  }
  /* Change `$TEXMFLS_R/' to `./'.  */
  if (FILESTRNCASEEQ(destdir, lsrdir, strlen(lsrdir)))
    relative_destdir = concat_pathes("./", destdir + strlen(lsrdir));
  else
    relative_destdir = destdir;
  fseek(f, 0L, SEEK_END);
  fprintf(f, "%s:\n%s\n", relative_destdir, name); 
  /*    fputs(destdir, f); fputs(":\n", f);
	fputs(name, f); fputc('\n', f); */
  fclose(f);
  /* update the current db too ! */
  destdirname = concat_pathes(destdir, name);
  kpse_db_insert(destdirname);
  free(relative_destdir);
  free(destdirname);
}

void maketexupdate(int argc, char **argv)
{
  do_maketexupdate(argv[1], argv[2]);
}

/* if dirname is a path that can expand into multiple direcotries,
   return the first directory that exists. */
string expand(const_string dirname)
{
  string path = kpse_path_expand(dirname);
  string elt;

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "MakeTeXMF (expand) <= %s\n\t <= %s\n", dirname, path);
  }

  for(elt = kpse_path_element(path); 
      elt && !test_file('d', elt); 
      elt = kpse_path_element(NULL)) {

    if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
      fprintf(stderr, "MakeTeXMF (expand) => %s\n", elt);
    }
  }
  return (elt != NULL ? xstrdup(elt) : NULL);
}

void do_maketexmf(string name, string *generated)
{
  string rootname, pointsize, realsize;
  string base, fontdir;
  string mfname, b_rootname;
  string p;
  FILE *tmpfile;

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MakeTeXMF:\n\tName: %s\n", name);
  }

  /* name = basename $1 .mf */
  name = xstrdup(basename(name));
  if (FILESTRCASEEQ(name+strlen(name)-3, ".mf"))
    *(name+strlen(name)-3) = '\0';
  /* rootname = `echo $name | sed 's/[0-9]*$//'` */
  rootname = xstrdup(name);
  for (p = rootname+strlen(rootname);  p > rootname && isdigit(p[-1]); p--);
  /* pointsize = `echo $name | sed 's/^$rootname//'` */
  pointsize = xstrdup(p);
  *p = '\0';
  
  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
    fprintf(stderr, "name = %s, rootname = %s, pointsize = %s\n",
	    name, rootname, pointsize);
  }      

  if (*pointsize == '\0') {
    fprintf(stderr, "%s: No pointsize.\n", progname);
    mt_exit(1);
  } else if (strncmp(pointsize, "11", 2) == 0)
    realsize = "10.95444";
  else if (strncmp(pointsize, "14", 2) == 0)
    realsize = "14.4";
  else if (strncmp(pointsize, "17", 2) == 0)
    realsize = "17.28";
  else if (strncmp(pointsize, "20", 2) == 0)
    realsize = "20.74";
  else if (strncmp(pointsize, "25", 2) == 0)
    realsize = "24.88";
  else if (strlen(pointsize) == 4) {
    /* realsize=`echo "$pointsize" | sed 's/\(..\)/\1./'` */
    realsize = xmalloc(6);
    realsize[0] = pointsize[0];
    realsize[1] = pointsize[1];
    realsize[2] = '.';
    realsize[3] = pointsize[2];
    realsize[4] = pointsize[3];
    realsize[5] = '\0';
  } else realsize = pointsize;

  /* FIXME: We have to expand fontdir and restrain it to the first 
     valid directory. */
  base = "";
  if (FILESTRNCASEEQ(name, "dc", 2)) {
    fontdir = expand(dcfontdir);
    base = "dxbase";
  }
  else if (FILESTRNCASEEQ(name, "ec", 2)) {
    fontdir = expand(ecfontdir);
    base = "exbase";
  }
  else if (FILESTRNCASEEQ(name, "tc", 2)) {
    fontdir = expand(tcfontdir);
    base = "exbase";
  }
  else {
    if (FILESTRNCASEEQ(name, "fc", 2))
      fontdir = expand(fcfontdir);
    else if (FILESTRNCASEEQ(name, "ts", 2))
      fontdir = expand(tsfontdir);
    else {
      /* first, look into sauterfontdir if it exists */
      fontdir = expand(sauterfontdir);
      if (fontdir && *fontdir != '\0') {
	/* does b-$(rootname).mf exist in sauterfontdir ? */
	/* FIXME: we can't free b-$rootname).mf */
	b_rootname = concat_pathes(fontdir, concat3("b-", rootname, ".mf"));
	if (!test_file('r', b_rootname)) {
	  fprintf(stderr, "%s: can't find %s.\n", progname, b_rootname);
	  mt_exit(1);
	}
	free(b_rootname);
      }
      else {
	string pathrootname;
	/* so , sauterfontdir doesn't exist, we might try to look for 
	   b-$(rootname) somewhere else */
	b_rootname = concat3("b-", rootname, ".mf");
	if ((pathrootname = kpse_find_file(b_rootname, kpse_mf_format, true)))
	  fontdir = dirname(pathrootname);
	else {
	  /* Fail : b-$(rootname) wasn't found */
	  fprintf(stderr, "%s: can't find %s.\n", progname, b_rootname);
	  mt_exit(1);
	}
	free(b_rootname);
      }
    }
  }

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MakeTeXMF:\n\tBase: %s; pointsize : %s\n", 
	    base, pointsize);
  }
  
  if (fontdir == NULL) {
    oops("%s: can't figure out fontname to build %s.mf!\n", progname, name);
    mt_exit(1);
  }
  pushd(fontdir);

  mfname = concat(name, ".mf");
  if (test_file('r', mfname)) {
    /* mfname exists already */
    fprintf(stderr, "%s: %s/%s already exists.\n", progname, fontdir, mfname);
    /* fputs(mfname, fout); fputc('\n', fout); */
  }
  else {
    if ((tmpfile = fopen(mfname, "w")) == NULL) {
      perror(mfname);
      mt_exit(1);
    }

    if (*base)
      fprintf(tmpfile, "if unknown %s: input %s fi;\n", base, base);
    else
      fprintf(tmpfile, "design_size := %s;\ninput b-%s;\n",
	      realsize, rootname);

    fprintf(tmpfile, "gensize := %s;\ngenerate %s;\n", realsize, rootname);
    fclose(tmpfile);
    chmod(mfname, file_perms);	/* probably, a no-op */
  }
  *generated = concat_pathes(fontdir, mfname);
  do_maketexupdate(fontdir, mfname);
}

void maketexmf(int argc, char **argv)
{
  string gen;
  do_maketexmf(argv[1], &gen);
  fputs(gen, fout); fputc('\n', fout);
  free(gen);
}

void maketextex(int argc, char **argv)
{

}

void do_maketextfm(string name, string dest, string *generated)
{
  string dpi = mt_bdpi;
  string pkbasename, tfmbasename;
  string gfname;
  string tfmdestdir, pkdestdir;
  string mode_var = kpse_var_value("MODE");
  char tfmtempname[PATH_MAX];
  char pktempname[PATH_MAX];
  char cmd[256];
  char errstring[PATH_MAX*4];
  int retcode;
#ifdef ENHANCED_MODE
  string mfname;
#endif
#ifdef MSDOS
  char *dot;
#endif

  name = xstrdup(basename(name));
  if (FILESTRCASEEQ(name+strlen(name) - 4, ".tfm"))
    *(name+strlen(name) - 4) = '\0';

#ifdef MSDOS
  /* Make sure to nuke any extension the file might have.  */
  if ((dot = strchr(name, '.')))
    *dot = '\0';
#endif
 
  /* Put files into proper place */
  do_maketexnames(name, dest, atoi(dpi), mode_var ? mode_var : NULL);
  pkbasename = (string)basename(pkname);
  tfmbasename = concat(name, ".tfm");
  tfmdestdir = mt_tfmdestdir;
  pkdestdir = mt_pkdestdir;

  /* FIXME: can e.g. cmr10.600gf clash with e.g. cmr10.600whatever on DOS?  */
  gfname = concatn(name, ".", dpi, "gf", NULL);

  if (KPSE_DEBUG_P(MAKETEX_FINE_DEBUG)) {
      fprintf(stderr, "tfmdestdir: %s\npkdestdir: %s\ntfmbasename: %s\npkbasename: %s\n", tfmdestdir, pkdestdir, tfmbasename, pkbasename);
      fprintf(stderr, "tfmname: %s\npkname: %s\n", tfmname, pkname);
    }

  /* does the tfm already exist ? */
  if (test_file('r', tfmname)) {
    fprintf(stderr, "%s: %s already exists.\n", progname, tfmname);
    /* fputs(tfmname, fout); fputc('\n', fout); */
    *generated = xstrdup(tfmname);
    do_maketexupdate(tfmdestdir, tfmbasename);
    return;
  }

#ifdef ENHANCED_MODE
  /* does the mf file exist ? */
  mfname = concat(name, ".mf");
  if (!kpse_find_file(mfname, kpse_mf_format, true)) {
    free(mfname);
    do_maketexmf(name, &mfname);
    /* Places were wrong in this case */
    do_maketexnames(name, dest, atoi(dpi), NULL);
  }
#endif

  do_makedir(tfmdestdir);

  /* Protect the semi-colons from being interpreted by the shell.
     (Use double quotes since they are supported on non-Unix platforms.)  */
  sprintf(cmd, "mf \"\\mode:=%s; mag:=1; scrollmode; input %s\"",
	  mt_mode, name);

  fprintf(stderr, "%s: Running %s\n", progname, cmd);

  /* Now run metafont.
     Note that stdin has been redirected to "/dev/null" by `main'. */
  retcode = system(cmd);
  if (retcode != 0) {
    if (retcode == -1)
      perror("mf");
    else if (!test_file('r', tfmbasename))
      fprintf(stderr, "%s: `%s' failed to make %s.\n",
	      progname, cmd, tfmbasename);
    mt_exit(1);
  }

  sprintf(tfmtempname, "%s/tfXXXXXX", tfmdestdir);
  mktemp(tfmtempname);
  if (mvfile(tfmbasename, tfmtempname) == FALSE) {
    sprintf(errstring, "%s: can't move %s to %s", progname, tfmbasename, 
	    tfmtempname);
    perror(errstring);
    mt_exit(1);
  }
  if (!test_file('r', tfmname))
    if (mvfile(tfmtempname, tfmname) == FALSE) {
      sprintf(errstring, "%s: can't move %s to %s", progname, tfmtempname, 
	      tfmname);
      perror(errstring);
      mt_exit(1);
    }
  chmod(tfmname, file_perms);	/* probably, a no-op */

  /* OK, success with the TFM.  */
  do_maketexupdate(tfmdestdir, tfmbasename);

  /* Maybe we should have been waiting before restoring stdin & stdout*/
  *generated = xstrdup(tfmname);

  /* Since we probably made a GF(->PK) file, too, may as well install
     it if it's needed.  */
  if (test_file('r', gfname) && !test_file('f', pkname)) {
    sprintf(cmd, "gftopk ./%s %s", gfname, pkbasename);
    if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
      fprintf(stderr, "Running: %s\n", cmd);
    }
    retcode = system(cmd);
    if (retcode != 0) {
      if (retcode == -1)
	perror("gftopk");
      else
	fprintf(stderr, "%s: gftopk failed to make %s.\n",
		progname, pkbasename);
      mt_exit(1);
    }
    
    do_makedir(pkdestdir);
    sprintf(pktempname, "%s/pkXXXXXX", pkdestdir);
    mktemp(pktempname);
    if (mvfile(pkbasename, pktempname) == FALSE) {
      sprintf(errstring, "%s: can't move %s to %s", progname, pkbasename, 
	      pktempname);
      perror(errstring);
      mt_exit(1);
    }
    if (test_file('f', pkname))
      remove(pktempname);
    else {
      if (mvfile(pktempname, pkname) == FALSE) {
	sprintf(errstring, "%s: can't move %s to %s", progname, pktempname, 
		pkname);
	perror(errstring);
	mt_exit(1);
      }
      chmod(pkname, file_perms);
      do_maketexupdate(pkdestdir, pkbasename);
    }
  }
}

/* this is similar to maketexpk ... */
void maketextfm(int argc, char **argv)
{
  string gen;
  do_maketextfm(argv[1], argv[2], &gen);
  fputs(gen, fout); fputc('\n', fout);
  free(gen);
}

#ifndef MSDOS
  
int get_mode_from_mf(string mode)
{
  int std_in[2], std_out[2];
  int new_fd[3], oldin, oldout, pid, status, result = 0;
  char buf[1024], *p;
  FILE *mf_in, *mf_out;

  if (pipe(std_in, 0, _O_TEXT) != 0 
      || pipe(std_out, 0, _O_TEXT) != 0) {
    perror("pipe");
    mt_exit(1);
  }
  /*
  _flushall();
  
  oldin = dup(0);
  oldout = dup(1);
  
  if (dup2(std_in[0], 0) != 0)
    perror("dup2 std_in");
  if (dup2(std_out[1], 1) != 0)
    perror("dup2 std_out");
  _flushall();
  */
  new_fd[0] = std_in[0];
  new_fd[1] = std_out[1];
  new_fd[2] = 2;
  push_fd(new_fd);

  pid = spawnlp(_P_NOWAIT, "mf", "mf", NULL);
  Sleep(50); /* Don't know why, but emacs is doing it too ... */

  /*
  dup2(oldin, 0);
  dup2(oldout, 1);
  */
  pop_fd();
  /* close(10); */
  /* close(12); */
  /*
  close(std_in[0]);
  close(std_out[1]);
  */

  if (pid == -1) {
    perror("Spawning mf.exe");
    close(std_in[1]); close(std_out[0]);
    mt_exit(1);
  }

  mf_in = fdopen(std_in[1], "w");
  mf_out = fdopen(std_out[0], "r");

  sprintf(buf, "\\mode:=%s; mode_setup; message\"BDPI= \"&decimal round pixels_per_inch;end.\n", mode);
  fprintf(mf_in, buf);
  _flushall();
  for(; result==0 && !feof(mf_out) ;) {
    memset(buf, '\0', 1024);
    fgets(buf, 1024, mf_out);
    for(p = buf; *p; p++)
      if (strncmp(p, "DPI= ", 5) ==0) {
	result=atoi(p+5);
	break;
      }
  }

  if (_cwait(&status, pid, 0) == -1) {
    perror("Waiting for mf.");
    mt_exit(1);
  }
  fclose(mf_in);
  fclose(mf_out);

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MetaFont found that mode %s has %d dpi.\n", mode, result);
  }

  return result;
}

#else  /* MSDOS */

/* MS-DOS doesn't have `pipe', so it can only use `popen'/`pclose'.
   Btw: why won't everybody else? we need to wait for MetaFont anyway.  */

int get_mode_from_mf(string mode)
{
  FILE *fpipe;
  char buf[1024];
  int result = 0;

  sprintf(buf, "mf \"\\mode:=%s;mode_setup;message\\\"BDPI= \\\"&decimal round pixels_per_inch;end.\"", mode);
  fpipe = popen(buf, "r");
  if (fpipe == NULL) {
    perror("pipe to mf");
    return 0;
  }

  /* awk '/DPI=/ {print $2}' */
  while (fgets(buf, sizeof buf, fpipe)) {
    char *p = strstr(buf, "DPI=");
    if (p)
      result = atoi(p + 5);	/* could break here, but AWK script doesn't */
  }
  pclose(fpipe);
  return result;
}

#endif /* MSDOS */

void maketexpk(int argc, char **argv)
{
  string name = argv[1];
  string dpi = argv[2];
  string bdpi = argv[3];
  string mag = argv[4];
  string mode = "";
  string dest = "";
  char line[256], cmd[256];
  string pkbasename;
  string pkdestdir;
  char gfname[PATH_MAX];
  char gfpkname[PATH_MAX];
  char pktempname[PATH_MAX];
  string cfg_name;
  FILE *config_file;
  string index;
  int len = strlen(name);
  int mf_bdpi, ibdpi;
  boolean res;
  int retcode;
  char errstring[PATH_MAX*4];

  if (argc >= 6)
    mode = argv[5];
  if (argc == 7)
    dest = argv[6];

  if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
    fprintf(stderr, "MakeTeXPK:\n\tName: %s\n\tDpi: %s\n\tBdpi: %s\n\tMag: %s\n\tMode: %s\n\tDest: %s\n", name, dpi, bdpi, mag, mode, dest);
  }

  /* look for name in psfont.map.
     Can't use `kpse_open_file' because it exits if the file doesn't
     exits, and we have a special exit procedure which cleans up.  */
  cfg_name = kpse_find_file("psfonts.map", kpse_dvips_config_format, false);
  if (cfg_name == NULL) oops("Cannot find file psfonts.map.");
  config_file = fopen(cfg_name, "r");
  if (config_file == NULL) oops("Cannot open file psfonts.map.");
  /* looking for pattern "^r?$NAME(0|8r)?([ \t]|$)" */
  res = false;
  while (fgets(line, sizeof(line) - 1, config_file) != NULL) {
    index = line;
    if (*line == 'r' && FILESTRNCASEEQ(line+1, name, len))
      index += len+1;
    else if (FILESTRNCASEEQ(line, name, len))
      index += len;
    else
      continue;
    if (*index == '0')
      index++;
    else if (*index == '8' && *(index+1) == 'r')
      index += 2;
    if (*index == '\0' || ISSPACE(*index)) {
      res = true;
      break;
    }
  }
  fclose(config_file);
  if (res == true) {
    mode = "modeless";
    sprintf(cmd, "%s %s %s", ps_to_pk, name, dpi);
  }
  else {
    if (strlen(mode) > 0) {
      if ((mf_bdpi = get_mode_from_mf(mode)) != atoi(bdpi)) {
	fprintf(stderr, "%s: Mismatched mode %s and resolution %s; ignoring mode.\n", progname, mode, bdpi);
	mode = "";
      }
    }
    if (strlen(mode) == 0  || strcmp(mode, "default") == 0) {
      ibdpi = atoi(bdpi);
      if (ibdpi == 300)
	mode = "cx";
      else if (ibdpi == 600)
	mode = "ljfour";
      else {
	fprintf(stderr, "%s: Can't guess mode for %s dpi devices.\n",
		progname, bdpi);
	fprintf(stderr, "%s: Use a config file, or update me.\n", progname);
	mt_exit(1);
      }
    }

    sprintf(cmd, "mf \"\\mode:=%s; mag:=%s; scrollmode; input %s\"",
	    mode, mag, name);
  }
    
  /* Put files into proper place */
  do_maketexnames(name, *dest ? dest : NULL, atoi(dpi), *mode ? mode : NULL);

  pkbasename = (string)basename(pkname);
  pkdestdir = mt_pkdestdir;

  /* FIXME: can foobar.DPIgf clash with foobar.<anything else> on MS-DOS?  */
  sprintf(gfname, "%s.%sgf", name, dpi);
  sprintf(gfpkname, "%s.%spk", name, dpi);

  if (test_file('e', pkname)) {
    fprintf(stderr, "%s: %s already exists.\n", progname, pkname);
    fputs(pkname, fout); fputc('\n', fout);
    do_maketexupdate(pkdestdir, pkbasename);
    return;
  }

  /* Now run metafont or gsftopk */
  do_makedir(pkdestdir);
  fprintf(stderr, "%s: Running %s\n", progname, cmd);
  retcode = system(cmd);

  if (retcode == -1)
    perror(mode);
  if (retcode != 0) {
    fprintf(stderr, "%s: `%s' failed.\n", progname, cmd);
    mt_exit(1);
  }
  
  if (test_file('r', gfname)) {
    sprintf(cmd, "gftopk ./%s %s", gfname, pkbasename);
    if (KPSE_DEBUG_P(MAKETEX_DEBUG)) {
      fprintf(stderr, "Running: %s\n", cmd);
    }
    retcode = system(cmd);
    if (retcode == -1)
      perror("gftopk");
    if (retcode != 0) {
      fprintf(stderr, "%s: gftopk failed to make %s.\n", progname, pkbasename);
      mt_exit(1);
    }
  }
  if (!test_file('f', pkbasename) && test_file('f', gfpkname))
    if (mvfile(gfpkname, pkbasename) == FALSE) {
      sprintf(errstring, "%s: can't move %s to %s", progname,
	      gfpkname, pkbasename);
      perror(errstring);
      mt_exit(1);
    }
  if (!test_file('s', pkbasename)) {
    fprintf(stderr, "%s: `%s' failed to make %s.\n", progname, cmd,
	    pkbasename);
    mt_exit(1);
  }

  sprintf(pktempname, "%s/pkXXXXXX", pkdestdir);
  mktemp(pktempname);
  if (mvfile(pkbasename, pktempname) == FALSE) {
    sprintf(errstring, "%s: can't move %s to %s", progname, pkbasename, 
	    pktempname);
    perror(errstring);
    mt_exit(1);
  }
  if (!test_file('r', pkname))
    if (mvfile(pktempname, pkname) == FALSE) {
      sprintf(errstring, "%s: can't move %s to %s", progname, pktempname, 
	      pkname);
      perror(errstring);
    mt_exit(1);
    }

  chmod(pkname, file_perms);
  do_maketexupdate(pkdestdir, pkbasename);

  /* Maybe we should have been waiting before restoring stdin & stdout*/
  fputs(pkname, fout); fputc('\n', fout);
}

/* This one should help to clean up a font tree in a TDS way. It
  should also regenerate any pk font older than the same tfm font. 
  */
void maketextdsify(int arc, char **argv)
{

}
