package org.apache.torque.util;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;

/**
 * A value plus its JDBC type.
 *
 * @version $Id: JdbcTypedValue.java 1448414 2013-02-20 21:06:35Z tfischer $
 */
public class JdbcTypedValue
{
    /** The JDBC type as in <code>java.sql.Types</code>. **/
    private int jdbcType;

    /** The value; may be null. */
    private Object value;

    /**
     * Constructor.
     *
     * @param jdbcType The JDBC type as in <code>java.sql.Types</code>.
     * @param value The value; may be null.
     */
    public JdbcTypedValue(Object value, int jdbcType)
    {
        this.jdbcType = jdbcType;
        this.value = value;
    }

    /**
     * Returns the JDBC type as in <code>java.sql.Types</code>.
     *
     * @return the JDBC type of the value.
     */
    public int getJdbcType()
    {
        return jdbcType;
    }

    /**
     * Sets the JDBC type as in <code>java.sql.Types</code>.
     *
     * @param jdbcType the JDBC type of the value.
     */
    public void setJdbcType(int jdbcType)
    {
        this.jdbcType = jdbcType;
    }

    /**
     * Returns the value.
     *
     * @return value the value, or null.
     */
    public Object getValue()
    {
        return value;
    }

    /**
     * Sets the value.
     *
     * @param value the value, may be null.
     */
    public void setValue(Object value)
    {
        this.value = value;
    }

    @Override
    public int hashCode()
    {
        HashCodeBuilder hashCodeBuilder = new HashCodeBuilder()
            .append(jdbcType)
            .append(value);
        return hashCodeBuilder.toHashCode();
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        JdbcTypedValue other = (JdbcTypedValue) obj;
        EqualsBuilder equalsBuilder = new EqualsBuilder()
            .append(jdbcType, other.jdbcType)
            .append(value, other.value);
        return equalsBuilder.isEquals();
    }

    @Override
    public String toString()
    {
        return "JdbcTypedValue [jdbcType=" + jdbcType
            + ", value=" + value + "]";
    }
}
